/* Adobe.Repeater.js -- Repeater behavior for Animal */
/*global define */

define(["lib/Zoot", "lib/tasks"],
function (Z, tasks) {
	"use strict";

	var	radians = Z.mathUtils.rad,
		globalRepeaterID = 1, repeaterContainerName = "RPTRArt";
	
	function getRepeaterMatrix(inCopyIndex, inAnchorPt, inPosition, inScale, inSkewAmount, inSkewAxis, inAngle)
	{
		var mtx = Z.Mat3(), scaleDiff = inScale, newScale;
		
		newScale = [Math.pow(scaleDiff[0], inCopyIndex), Math.pow(scaleDiff[1], inCopyIndex)];
		
		mtx = mtx.setIdentity().translate(inPosition.scale(inCopyIndex))
											.rotate(radians(inAngle)*inCopyIndex)
											.scale(newScale)
											.skew(radians(inSkewAmount * inCopyIndex), radians(inSkewAxis * inCopyIndex))
											.translate(inAnchorPt.scale(inCopyIndex));
		
		return mtx;
	}
	
	return {
		about: "$$$/private/animal/Behavior/Repeater/About=Repeater",
		description: "$$$/private/animal/Behavior/Repeater/Desc=Repeating Transformer",
		uiName: "$$$/private/animal/Behavior/Repeater/UIName=Repeater",
		defaultArmedForRecordOn: true,
		hideInBehaviorList: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{ id: "midiInput", type: "eventGraph", uiName: "$$$/private/animal/Behavior/Repeater/Parameter/midiInput=MIDI Input",
					inputKeysArray: ["Midi/"], uiToolTip: "$$$/private/animal/Behavior/Repeater/Parameter/midiInput/tooltip=MIDI input", defaultArmedForRecordOn: true
				},
				{id:"midiChannel", type:"slider", 	uiName:"$$$/private/animal/Behavior/Repeater/Param/MidiChannel=Midi Channel",	precision:0, dephault:1,
																		min:1, max:16},

				{id:"numRepeats", type:"slider", 	uiName:"$$$/private/animal/Behavior/MidiFun/Param/NumRepeats=Copies",	precision:0, dephault:3,
																		min:0},
				{id:"copyOffset", type:"slider", 	uiName:"$$$/private/animal/Behavior/MidiFun/Param/CopyOffset=Copy Offset",	precision:1, dephault:0},
				{id:"copyOffsetVelocity", type:"slider", 	uiName:"$$$/private/animal/Behavior/MidiFun/Param/CopyOffsetVelocity=Copy Offset Velocity",	precision:1, dephault:0},
				{id:"anchorPtX", type:"slider", 	uiName:"$$$/private/animal/Behavior/Repeater/Param/AnchorPointX=Anchor Point X",	precision:1, dephault:0},
				{id:"anchorPtY", type:"slider", 	uiName:"$$$/private/animal/Behavior/Repeater/Param/AnchorPointY=Anchor Point Y",	precision:1, dephault:0},
				{id:"positionX", type:"slider", 	uiName:"$$$/private/animal/Behavior/Repeater/Param/PositionX=Position X",	precision:1, dephault:0},
				{id:"positionY", type:"slider", 	uiName:"$$$/private/animal/Behavior/Repeater/Param/PositionY=Position Y",	precision:1, dephault:0},
				{id:"uniformScale", type:"slider",	uiName:"$$$/private/animal/Behavior/Repeater/Param/Scale=Scale",		precision:1, uiUnits:"%", dephault:100},
				{id:"scaleX", type:"slider",		uiName:"$$$/private/animal/Behavior/Repeater/Param/ScaleX=Scale X",	precision:1, uiUnits:"%", dephault:100},
				{id:"scaleY", type:"slider",		uiName:"$$$/private/animal/Behavior/Repeater/Param/ScaleY=Scale Y",	precision:1, uiUnits:"%", dephault:100},
				{id:"skew", type:"slider",			uiName:"$$$/private/animal/Behavior/Repeater/Param/Skew=Skew",			min:-85, max:85, precision:1, dephault:0, hidden : true},
				{id:"skewAxis", type:"angle", 		uiName:"$$$/private/animal/Behavior/Repeater/Param/SkewAxis=Skew Axis",	precision:0, dephault:0, hidden : true},
				{id:"rotation", type:"angle",		uiName:"$$$/private/animal/Behavior/Repeater/Param/Rotation=Rotation", precision:0, dephault:0},
				{id:"opacity", type:"slider",		uiName:"$$$/private/animal/Behavior/Repeater/Param/Opacity=Opacity", min:0, max:100, precision:0, dephault:100, uiUnits:"%"}
			];
		},

		onResetRehearsalData : function (self,args) {
			self.lastPhase = 0.0;
			delete self.lastPhaseTime;	
		},
		
		onCreateBackStageBehavior: function (self) {
			return { order: 0.99, importance : 0.0 };
		},

		onCreateStageBehavior: function (self, args) {
			self.repeaterID = globalRepeaterID++;
			self.builder = Z.nml.newBuilder();

			// initialize the rehearsal data.
			this.onResetRehearsalData(self);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage		
			var rootContainer = args.stagePuppet.getParentLayer().getDisplayContainer(), art, 
				midiChannel = args.getParam("midiChannel"), numRepeats = args.getParam("numRepeats"),
				anchorPt = Z.Vec2([-args.getParam("anchorPtX"), -args.getParam("anchorPtY")]),
				position = Z.Vec2([args.getParam("positionX"), args.getParam("positionY")]),
				uniformScale = args.getParam("uniformScale") * 0.01,
				scale = Z.Vec2([args.getParam("scaleX") * 0.01, args.getParam("scaleY") * 0.01]),
				skewAmount = args.getParam("skew"),
				skewAxis = args.getParam("skewAxis"),
				angle = args.getParam("rotation"),
				copyOffset = args.getParam("copyOffset"),
				copyOffsetVelocity = args.getParam("copyOffsetVelocity"),
				copyPhase, 
				opacity = args.getParam("opacity") * 0.01, copyOpacity, copyIndex,
				mtx, cc, children, rptrContainer0, i, j, sourceArt, t;

			scale = scale.scale(uniformScale);
			cc = args.getParamEventValue("midiInput", "Midi/1/CC/16/Value");
			if (cc === undefined) {
				cc = 1;
			} else {
				cc /= 128.0;
			}
			
			children = rootContainer.getChildren();
			
			for (j = 0; j < children.length; j += 1) {
				if (children[j].getName() === repeaterContainerName) {
					rptrContainer0 = children[j];
					rootContainer.removeChildAtIndex(j);
					break;
				}
			}
			
			if (rptrContainer0) {
				if (rptrContainer0.lastRenderT === args.t && 
				   	rptrContainer0.lastRenderGlobalRehearseTime === args.globalRehearseTime &&
				   	rptrContainer0.repeaterID !== self.repeaterID) {
					// Repeating another repeater
					sourceArt = rptrContainer0.clone();
				} else {
					sourceArt = rootContainer.clone();
					children = sourceArt.getChildren();
					for (j = 0; j < children.length; j += 1) {
						children[j].setAlpha(1.0);
					}
				}
				rptrContainer0.removeChildren();
			} else {
				sourceArt = rootContainer.clone();
			}
			
			children = rootContainer.getChildren();
			for (j = 0; j < children.length; j += 1) {
				children[j].setAlpha(0.0);
			}
			
			rptrContainer0 = self.builder.newContainer(repeaterContainerName);
			rptrContainer0.lastRenderT= args.t;
			rptrContainer0.lastRenderGlobalRehearseTime = args.globalRehearseTime;
			rptrContainer0.repeaterID = self.repeaterID;

			t = args.t + args.globalRehearseTime;
			if (self.lastPhaseTime === undefined) {
				self.lastPhaseTime = t;
			}
			self.lastPhase += (self.lastPhaseTime - t) * copyOffsetVelocity * cc;
				
			copyPhase = (self.lastPhase) % 1.0;
			if (copyPhase < 0) copyPhase++;

			self.lastPhaseTime = t;
			
			for (i = -1; i < numRepeats; i = i + 1) 
			{
				copyIndex = i + copyPhase;
				if (copyIndex <= -1 || copyIndex >= numRepeats) continue;

				art = sourceArt.clone();
				copyOpacity = opacity;
				
				if (copyIndex < 0) {
					copyOpacity *= 1 + copyIndex;
				} else if ((copyIndex+1) > numRepeats) {
					copyOpacity *= numRepeats - copyIndex;
				}
				art.setAlpha(copyOpacity);
				
				mtx = getRepeaterMatrix(copyIndex + copyOffset, anchorPt, position, scale, skewAmount, skewAxis, angle);
				art.setMatrix(mtx);
				
				rptrContainer0.addChild(art);
			}
			rootContainer.addChild(rptrContainer0, 0);
			args.setEventGraphParamRecordingValid("midiInput");
		}

	}; // end of object being returned
});
